import LicenseSpring

import SwiftUI

private let LICENSESPRINGAPIKEY = ""
private let LICENSESPRINGSHAREDKEY = ""
private let LICENSESPRINGCLIENTID = ""
private let LICENSESPRINGCLIENTSECRET = ""
private let LICENSESPRINGCRYPTOPROVIDERKEY = ""
private let LICENSESPRINGPRODUCTSHORTCODE = ""

enum ManagerType: Hashable, Codable {
    case licenseManager
    case floatingClient
}

struct ConfigView: View {
    @Binding var managerType: ManagerType
    let onComplete: (Configuration?) throws -> Void
    @State private var apiKey = ""
    @State private var sharedKey = ""
    @State private var productCode = ""
    @State private var accountCode = ""
    @State private var airgapKey = ""
    @State private var clientId = ""
    @State private var clientSecret = ""
    @State private var cryptoProviderKey = ""
    @State private var floatingServerURL = ""
    @State private var errorText: String?
    @State private var enableVM: Bool = false

    var body: some View {
        VStack {
            Picker("", selection: $managerType) {
                Text("LicenseManager").tag(ManagerType.licenseManager)
                Text("FloatingClient").tag(ManagerType.floatingClient)
            }
            .labelsHidden()
            .pickerStyle(.segmented)
            Form {
                if managerType == .licenseManager {
                    TextField("API Key", text: $apiKey)
                        .lineLimit(1)
                    TextField("Shared Key", text: $sharedKey)
                        .lineLimit(1)
                    TextField("Client ID", text: $clientId)
                        .lineLimit(1)
                    TextField("Client Secret", text: $clientSecret)
                        .lineLimit(1)
                    TextField("Crypto Provider Key", text: $cryptoProviderKey)
                        .lineLimit(1)
                }
                TextField("Product Code", text: $productCode)
                    .lineLimit(1)
                if managerType == .licenseManager {
                    TextField("Account Code (SSO)", text: $accountCode)
                        .lineLimit(1)
                    TextField("Air-gap Key", text: $airgapKey)
                        .lineLimit(1)
                }
                if managerType == .floatingClient {
                    TextField("Floating Server URL", text: $floatingServerURL)
                        .lineLimit(1)
                }
                Toggle(isOn: $enableVM){
                    Text("Enable VM")
                }
            }
            if let errorText {
                Text(errorText)
                    .frame(maxWidth: .infinity, alignment: .trailing)
                    .font(.caption)
                    .foregroundColor(Color.red)
            }
            HStack {
                Spacer()
                MinWidthButton("Exit") {
                    try? onComplete(nil)
                }
                MinWidthButton("Configure") {
                    do {
                        let config = try currentConfiguration
                        try onComplete(config)
                        
                        UserDefaults.standard.set(apiKey, forKey: "CachedAPIKey")
                        UserDefaults.standard.set(sharedKey, forKey: "CachedSharedKey")
                        UserDefaults.standard.set(productCode, forKey: "CachedProductCode")
                        UserDefaults.standard.set(clientId, forKey: "CachedClientId")
                        UserDefaults.standard.set(clientSecret, forKey: "CachedClientSecret")
                        UserDefaults.standard.set(cryptoProviderKey, forKey: "CachedCryptoProviderKey")
                        UserDefaults.standard.set(accountCode, forKey: "CachedAccountCode")
                        UserDefaults.standard.set(airgapKey, forKey: "CachedAirGapPublicKey")
                        UserDefaults.standard.set(floatingServerURL, forKey: "CachedFloatingServerURL")
                        
                        errorText = nil
                    } catch {
                        errorText = error.localizedDescription
                    }
                }
                .disabled(!configReady)
            }
        }
        .onAppear { reloadCachedFields() }
        .valueChanged(value: managerType) { _ in reloadCachedFields() }
    }
    
    private func reloadCachedFields() {
        productCode = UserDefaults.standard.string(forKey: "CachedProductCode") ?? LICENSESPRINGPRODUCTSHORTCODE
        accountCode = UserDefaults.standard.string(forKey: "CachedAccountCode") ?? ""
        airgapKey = UserDefaults.standard.string(forKey: "CachedAirGapPublicKey") ?? ""
        apiKey = ""
        sharedKey = ""
        floatingServerURL = ""
        
        switch managerType {
        case .licenseManager:
            apiKey = UserDefaults.standard.string(forKey: "CachedAPIKey") ?? LICENSESPRINGAPIKEY
            sharedKey = UserDefaults.standard.string(forKey: "CachedSharedKey") ?? LICENSESPRINGSHAREDKEY
            clientId = UserDefaults.standard.string(forKey: "CachedClientId") ?? LICENSESPRINGCLIENTID
            clientSecret = UserDefaults.standard.string(forKey: "CachedClientSecret") ?? LICENSESPRINGCLIENTSECRET
            cryptoProviderKey = UserDefaults.standard.string(forKey: "CachedCryptoProviderKey") ?? LICENSESPRINGCRYPTOPROVIDERKEY
        case .floatingClient:
            floatingServerURL = UserDefaults.standard.string(forKey: "CachedFloatingServerURL") ?? ""
        }
    }
    
    private var configReady: Bool {
        (!apiKey.isEmpty && !sharedKey.isEmpty && !productCode.isEmpty) || (isOAuth)
    }
    
    private var isOAuth: Bool {
        !clientId.isEmpty && !clientSecret.isEmpty && !cryptoProviderKey.isEmpty && !productCode.isEmpty
    }
    
    private var currentConfiguration: Configuration? {
        get throws {
            // If the instance is created with a licenseFilePath to nil, the license will be located in Application Support /name_of_the_app/
            let licenseFilePath: String? = nil // use default license file path, but you can provide your desired path
            let serviceURL: String? = nil //you can use default service url, but you can also use your custom url
            let hardwareID: String? = nil // use default HardwareID, but you can calculate device fingerprint by your own
            var configuration: Configuration
            let extendedOptions = ExtendedOptions(enableVMDetection: enableVM)
            if (isOAuth){
                let oAuthCreds = try OAuthCredentials(clientId: clientId, clientSecret: clientSecret)
                oAuthCreds.cryptoProviderKey = cryptoProviderKey
                configuration = Configuration(oAuthCredentials: oAuthCreds, productCode: productCode.normalized, extendedOptions: extendedOptions)
            }
            else {
                configuration = Configuration(apiKey: apiKey.normalized, sharedKey: sharedKey.normalized, productCode: productCode.normalized, extendedOptions: extendedOptions)
            }
            configuration.appName = "Custom app"
            configuration.appVersion = "1.0.0"
            if let licenseFilePath = licenseFilePath {
                let file = URL(fileURLWithPath: licenseFilePath)
                configuration.licenseDirectoryPath = file.deletingLastPathComponent()
                configuration.licenseFileName = file.lastPathComponent
            }
            if let serviceURL = serviceURL.flatMap(URL.init(string:)) {
                configuration.serviceURL = serviceURL
            }
            if let hardwareID = hardwareID {
                configuration.hardwareID = hardwareID
            }
            configuration.collectHostNameAndLocalIP = true
            configuration.networkTimeout = 30
            
            if !accountCode.isEmpty {
                configuration.customerAccountCode = accountCode // for SSO support
            }
            if !floatingServerURL.isEmpty, let url = URL(string: floatingServerURL) {
                configuration.serviceURL = url
            }
            if !airgapKey.isEmpty {
                configuration.airgapKey = airgapKey // for air-gap support
            }
            
            return configuration
        }
    }
}

extension String {
    fileprivate var normalized: String {
        replacingOccurrences(of: " ", with: "").replacingOccurrences(of: "\n", with: "")
    }
}

struct ConfigView_Previews: PreviewProvider {
    static var previews: some View {
        ConfigView(managerType: .constant(.licenseManager)) { _ in }
    }
}
